package gc;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Scanner;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Georgiy Korneev (kgeorgiy@kgeorgiy.info)
 */
public final class GCShell {
    private GCShell() {
    }

    private static final List<GC> COLLECTORS = List.of(
            new GC("Serial", "Serial GC", "-XX:+UseSerialGC"),
            new GC("Parallel", "Parallel GC", "-XX:+UseParallelGC"),
            new GC("G1", "Garbage First GC", "-XX:+UseG1GC"),
            new GC("Shenandoah", "Shenandoah", "-XX:+UseShenandoahGC"),
            new GC("Z", "Z GC", "-XX:+UseZGC"),
            new GC("Epsilon", "Epsilon GC", "-XX:+UnlockExperimentalVMOptions", "-XX:+UseEpsilonGC")
    );

    public static void main(final String[] args) throws IOException, InterruptedException {
        final Scanner in = new Scanner(System.in);
        while (true) {
            System.out.println("\n\nChoose garbage collector");
            for (final GC gc : COLLECTORS) {
                System.out.printf("[%s]%-15s - %s %s%n", gc.name.charAt(0), gc.name.substring(1), gc.description, gc.options);
            }

            if (!in.hasNextLine()) {
                break;
            }
            final Scanner line = new Scanner(in.nextLine().strip());
            if (!line.hasNext()) {
                break;
            }

            final String command = line.next().toUpperCase();
            boolean found = false;
            for (final GC gc : COLLECTORS) {
                if (command.startsWith(gc.name.substring(0, 1))) {
                    found = true;
                    run(gc);
                }
            }
            if (!found) {
                System.out.format("Unknown command %s%n", command);
            }
        }
    }

    private static String getClassPath(final Class<?>... dependencies) {
        return Arrays.stream(dependencies)
                .map(dependency -> {
                    try {
                        return Path.of(dependency.getProtectionDomain().getCodeSource().getLocation().toURI()).toString();
                    } catch (final URISyntaxException e) {
                        throw new AssertionError(e);
                    }
                })
                .collect(Collectors.joining(File.pathSeparator));
    }

    private static void run(final GC gc) throws IOException, InterruptedException {
        System.out.format("=== Running %s%n", gc.description);
        final List<String> command = Stream.of(
                Stream.of("java", "-Xlog:gc*", "-Xmx1024m", "--class-path", getClassPath(GCDemo.class)),
                gc.options.stream(),
                Stream.of(GCDemo.class.getName())
        ).flatMap(Function.identity()).toList();
        System.out.println(String.join(" ", command));
        final Process process = new ProcessBuilder(command)
                .inheritIO()
                .start();
        final int exitCode = process.waitFor();
        System.out.format("=== Finished %s (exit code %d)%n", gc.description, exitCode);
    }

    record GC(String name, String description, List<String> options) {
        GC(final String name, final String description, final String... options) {
            this(name, description, List.of(options));
        }
    }
}
